import express from "express";
import { execFile } from "node:child_process";
import fs from "node:fs";
import path from "node:path";

const SOCKET_PATH = process.env.USBIP_AGENT_SOCKET || "/run/usbip-webui/agent.sock";
const HOST_RE = /^[A-Za-z0-9.:_-]{1,253}$/;
const BUSID_RE = /^[0-9]+-[0-9]+(?:\.[0-9]+)*$/;

function httpErr(code, msg) {
  const e = new Error(msg);
  e.statusCode = code;
  return e;
}

function validateHost(s) {
  s = (s || "").trim();
  if (!s) throw httpErr(400, "host is required");
  if (!HOST_RE.test(s)) throw httpErr(400, "invalid host");
  return s;
}
function validateBusid(b) {
  b = (b || "").trim();
  if (!BUSID_RE.test(b)) throw httpErr(400, "invalid busid");
  return b;
}
function validateTcpPort(p) {
  if (p === undefined || p === null || p === "") return 3240;
  const n = Number(p);
  if (!Number.isFinite(n) || n < 1 || n > 65535) throw httpErr(400, "invalid tcpPort");
  return Math.trunc(n);
}

function runCmd(argv, timeoutMs = 20000) {
  return new Promise((resolve, reject) => {
    execFile(argv[0], argv.slice(1), { timeout: timeoutMs }, (err, stdout, stderr) => {
      if (err) {
        const msg = (String(stderr || stdout || err.message || "Command failed")).trim();
        reject(httpErr(500, msg));
        return;
      }
      resolve({ stdout: String(stdout), stderr: String(stderr) });
    });
  });
}

function usbipArgv(tcpPort, command, args) {
  const port = validateTcpPort(tcpPort);
  if (port && port !== 3240) return ["usbip", "--tcp-port", String(port), command, ...args];
  return ["usbip", command, ...args];
}

async function usbipPortRaw() {
  const { stdout } = await runCmd(["usbip", "port"], 15000);
  return stdout;
}

async function resolvePortByBusid(busid) {
  busid = validateBusid(busid);
  const out = await usbipPortRaw();
  const lines = out.split("\n");
  let currentPort = null;
  for (const ln of lines) {
    const m = ln.match(/^\s*Port\s+(\d+)\s*:/i);
    if (m) currentPort = Number(m[1]);
    const rb = ln.match(/Remote Bus ID:\s*([0-9]+-[0-9]+(?:\.[0-9]+)*)/i);
    const lb = ln.match(/Local Bus ID:\s*([0-9]+-[0-9]+(?:\.[0-9]+)*)/i);
    const b = rb?.[1] || lb?.[1];
    if (b && currentPort !== null && b === busid) return currentPort;
  }
  throw httpErr(404, "Device not attached (busid not found in usbip port)");
}

async function ensureVhci() {
  await runCmd(["modprobe", "vhci_hcd"], 10000).catch(() => {});
}

const app = express();
app.use(express.json());

app.get("/health", (req, res) => {
  res.json({ ok: true, pid: process.pid, uid: process.getuid?.() });
});

app.get("/usbip/port", async (req, res, next) => {
  try {
    const stdout = await usbipPortRaw();
    res.json({ stdout });
  } catch (e) { next(e); }
});

app.post("/usbip/list", async (req, res, next) => {
  try {
    const host = validateHost(req.body?.host);
    const tcpPort = validateTcpPort(req.body?.tcpPort);
    const argv = usbipArgv(tcpPort, "list", ["--remote", host]);
    const { stdout } = await runCmd(argv, 25000);
    res.json({ stdout });
  } catch (e) { next(e); }
});

app.post("/usbip/attach", async (req, res, next) => {
  try {
    const host = validateHost(req.body?.host);
    const busid = validateBusid(req.body?.busid);
    const tcpPort = validateTcpPort(req.body?.tcpPort);
    await ensureVhci();
    const argv = usbipArgv(tcpPort, "attach", ["--remote", host, "--busid", busid]);
    await runCmd(argv, 30000);
    const stdout = await usbipPortRaw();
    res.json({ ok: true, stdout });
  } catch (e) { next(e); }
});

app.post("/usbip/detach", async (req, res, next) => {
  try {
    const busid = (req.body?.busid || "").toString().trim();
    let port = req.body?.port;
    if (busid) port = await resolvePortByBusid(busid);

    const n = Number(port);
    if (!Number.isFinite(n) || n < 0 || n > 255) throw httpErr(400, "Invalid port");
    await runCmd(["usbip", "detach", "--port", String(Math.trunc(n))], 30000);
    const stdout = await usbipPortRaw();
    res.json({ ok: true, stdout });
  } catch (e) { next(e); }
});

app.use((err, req, res, next) => {
  const code = err?.statusCode || 500;
  res.status(code).json({ ok: false, detail: String(err?.message || err) });
});

function ensureDirForSocket(sockPath) {
  const dir = path.dirname(sockPath);
  fs.mkdirSync(dir, { recursive: true });
}

function start() {
  ensureDirForSocket(SOCKET_PATH);
  try { fs.unlinkSync(SOCKET_PATH); } catch {}
  const server = app.listen(SOCKET_PATH, () => {
    try { fs.chmodSync(SOCKET_PATH, 0o660); } catch {}
    console.log(`USB/IP WebUI Agent listening on unix:${SOCKET_PATH}`);
  });
  server.on("error", (e) => {
    console.error("Agent error:", e);
    process.exit(1);
  });
}
start();
