
let __refreshExportsTimer = null;
function scheduleExportsRefresh(delayMs = 500) {
  if (__refreshExportsTimer) return;
  __refreshExportsTimer = setTimeout(async () => {
    __refreshExportsTimer = null;
    try { await requestExports(true); } catch {}
  }, delayMs);
}
const $ = (id) => document.getElementById(id);

function log(line) {
  const el = $("log");
  el.textContent += line + "\n";
  el.scrollTop = el.scrollHeight;
}

function logK(key, vars = {}) { log(t(key, vars)); }


function setWsIndicator(state) {
  const dot = document.getElementById('wsDot');
  if (!dot) return;
  dot.classList.remove('wson','wsconnect','wsoff');
  if (state === 'on') dot.classList.add('wson');
  else if (state === 'connect') dot.classList.add('wsconnect');
  else dot.classList.add('wsoff');
}

function setStatus(_text) { /* deprecated */ }

async function api(path, opts = {}) {
  const res = await fetch(path, {
    headers: { "Content-Type": "application/json" },
    ...opts,
  });
  const text = await res.text();
  let data;
  try { data = text ? JSON.parse(text) : {}; } catch { data = { raw: text }; }
  if (!res.ok) {
    const detail = data?.detail || data?.raw || ("HTTP " + res.status);
    throw new Error(detail);
  }
  return data;
}

function escapeHtml(s) {
  return String(s ?? "")
    .replaceAll("&", "&amp;")
    .replaceAll("<", "&lt;")
    .replaceAll(">", "&gt;")
    .replaceAll('"', "&quot;")
    .replaceAll("'", "&#039;");
}


let ws = null;

function connectWS() {
  const proto = (location.protocol === "https:") ? "wss" : "ws";
  const url = `${proto}://${location.host}/ws`;
  setWsIndicator('connect');
  ws = new WebSocket(url);

  ws.onopen = () => {
        setWsIndicator("on");
    try { ws.send(JSON.stringify({ type: "getState" })); } catch {}
  };

  ws.onmessage = (ev) => {
    let msg = null;
    try { msg = JSON.parse(ev.data); } catch { return; }

    if (msg.type === "log" && msg.line) {
      log(msg.line);
      return;
    }

    if (msg.type === "servers") {
      state.servers = msg.servers || state.servers;
      if (msg.selectedServerId !== undefined) state.selectedServerId = msg.selectedServerId || state.selectedServerId;
      if (msg.autoConnectIntervalSec !== undefined) state.autoConnectIntervalSec = Number(msg.autoConnectIntervalSec) || state.autoConnectIntervalSec;
      renderServers();
      renderSettings();
      return;
    }

    if (msg.type === "state") {
      state.servers = msg.servers || [];
      state.selectedServerId = msg.selectedServerId || (state.servers[0]?.id || "");      state.autoConnectIntervalSec = Number(msg.autoConnectIntervalSec) || 5;
      state.attachedBusids = msg.attachedBusids || [];
      renderServers();
      renderSettings();
      // render with cached exports if any
      renderExports(window.__lastExports || []);
      return;
    }

    if (msg.type === "exports") {
      window.__lastExports = msg.devices || [];
      state.exportsDevices = window.__lastExports;
      renderExports(window.__lastExports);
      return;
    }

    if (msg.type === "ports") {
      const prev = new Set((state.attachedBusids || []).map(String));
      const nextArr = (msg.attachedBusids || []).map(String);
      const next = new Set(nextArr);

      // Detect unexpected drops (e.g., remote server closed the USB/IP connection)
      for (const b of prev) {
        if (!next.has(b)) {
          if (pendingUserDetaches.has(b)) { pendingUserDetaches.delete(b); }
          else { logK("log.portDropped", { busid: b }); scheduleExportsRefresh(250); }
        }
      }

      state.attachedBusids = nextArr;
      if (window.__lastExports) renderExports(window.__lastExports);
      return;
    }
  };

  ws.onclose = () => {
        setWsIndicator("off");
    setTimeout(connectWS, 1500);
  };
}

let state = {
  servers: [],
  selectedServerId: "",  autoConnectIntervalSec: 5,
  autoTimer: null,
  attachedBusids: [],
  ports: [],
  exportsDevices: []
};

// Tracks user-initiated detaches so we don't log them as unexpected drops.
let pendingUserDetaches = new Set();


function selectedServer() {
  return state.servers.find(s => s.id === state.selectedServerId) || null;
}

// ------------------- i18n -------------------
let LANG = "en";
let STATUS_KEY = "status.ready";

function t(key, vars = {}) {
  const dict = (window.I18N && window.I18N[LANG]) ? window.I18N[LANG] : (window.I18N?.en || {});
  let s = dict[key] || (window.I18N?.en?.[key]) || key;
  for (const [k, v] of Object.entries(vars)) {
    s = s.replaceAll(`{${k}}`, String(v));
  }
  return s;
}

function applyI18n() {
  document.documentElement.lang = LANG;

  document.querySelectorAll("[data-i18n]").forEach((el) => {
    el.textContent = t(el.getAttribute("data-i18n"));
  });

  document.querySelectorAll("[data-i18n-placeholder]").forEach((el) => {
    el.setAttribute("placeholder", t(el.getAttribute("data-i18n-placeholder")));
  });

  document.querySelectorAll("[data-i18n-title]").forEach((el) => {
    el.setAttribute("title", t(el.getAttribute("data-i18n-title")));
  });

  updateStatusDot();

  if (window.__lastExports) renderExports(window.__lastExports);
}

function setLang(lang) {
  LANG = (lang === "es") ? "es" : "en";
  try { localStorage.setItem("usbipWebUiLang", LANG); } catch {}
  const sel = $("langSelect");
  if (sel) sel.value = LANG;
  applyI18n();
}

function updateStatusDot() {
  const dot = $("statusDot");
  if (!dot) return;
  dot.classList.remove("stok", "stbusy", "sterr");
  const k = STATUS_KEY || "status.ready";
  if (k === "status.error") dot.classList.add("sterr");
  else if (k === "status.ready") dot.classList.add("stok");
  else dot.classList.add("stbusy");
}

function setStatusKey(key) {
  STATUS_KEY = key || "status.ready";
  updateStatusDot();
}


function getSelectedServer() { return selectedServer(); }

function badgeForStatus(status) {
  const s = (status || "").toLowerCase();
  if (s.includes("in use") || s.includes("used") || s.includes("attached")) return `<span class="badge ok">${escapeHtml(t("ports.inUse"))}</span>`;
  if (s.includes("available")) return `<span class="badge">${escapeHtml(t("ports.free"))}</span>`;
  return `<span class="badge warn">?</span>`;
}

function renderServers() {
  const sel = $("serverSelect");
  sel.innerHTML = "";

  if (!state.servers.length) {
    const opt = document.createElement("option");
    opt.value = "";
    opt.textContent = t("servers.emptyOption");
    sel.appendChild(opt);
    return;
  }

  for (const s of state.servers) {
    const opt = document.createElement("option");
    opt.value = s.id;
    const hp = (s.tcpPort && Number(s.tcpPort) !== 3240) ? `${s.host}:${s.tcpPort}` : s.host;
    const label = s.name ? `${s.name} (${hp})` : hp;
    opt.textContent = label;
    if (s.id === state.selectedServerId) opt.selected = true;
    sel.appendChild(opt);
  }
}

function renderSettings() {
  $("autoInterval").value = String(state.autoConnectIntervalSec || 5);
}

function renderPorts(ports) {
  // UI simplified: we no longer show a separate 'local ports' table.
  // We still keep ports in memory to show attached state in the exports list.
  const inUse = (ports || []).filter(p => /in use/i.test(p.status || "")).length;
  const el = document.getElementById("portsCount");
  if (el) el.textContent = String(inUse);
}

function isBusidAttached(busid) {
  const list = state.attachedBusids || [];
  return list.includes(busid);
}


function renderExportsMessage(key, vars = {}) {
  const el = $("exportsTbody");
  if (!el) return;
  const msg = t(key, vars);
  el.innerHTML = `<tr><td colspan="5" class="muted">${escapeHtml(msg)}</td></tr>`;
}
function renderExports(devices) {
  const sel = getSelectedServer();
  const autoList = (sel?.autoVidPids || []).map(String).map(v => v.toLowerCase());

  const rows = (devices || []).map((d) => {
    const busid = d.busid;
    const vp = (d.vid && d.pid) ? `${String(d.vid).toLowerCase()}:${String(d.pid).toLowerCase()}` : "";
    const connected = isBusidAttached(busid);
    const auto = vp ? autoList.includes(vp) : false;

    const autoCell = vp
      ? `<label class="chk" title="${escapeHtml(t("devices.autoTitle", { vp }))}"><input type="checkbox" data-vidpid="${escapeHtml(vp)}" ${auto ? "checked" : ""}/> <span></span></label>`
      : `<span class="muted small" title="${escapeHtml(t("devices.unknownVidpidTitle"))}">—</span>`;

    const btn = connected
      ? `<button class="danger" data-detach="${escapeHtml(busid)}">${escapeHtml(t("devices.disconnect"))}</button>`
      : `<button class="primary" data-attach="${escapeHtml(busid)}">${escapeHtml(t("devices.connect"))}</button>`;

    const vidpidText = vp || "—";

    return `<tr>
      <td class="mono">${escapeHtml(busid)}</td>
      <td>${escapeHtml(d.desc || "")}</td>
      <td class="mono">${escapeHtml(vidpidText)}</td>
      <td>${autoCell}</td>
      <td>${btn}</td>
    </tr>`;
  }).join("");

  const el = $("exportsTbody");
  if (!el) return;
  el.innerHTML = rows || `<tr><td colspan="5" class="muted">${escapeHtml(t("devices.none"))}</td></tr>`;

  // bind attach/detach
  el.querySelectorAll("button[data-attach]").forEach((b) => {
    b.addEventListener("click", async () => {
      const busid = b.getAttribute("data-attach");
      await attach(busid);
    });
  });
  el.querySelectorAll("button[data-detach]").forEach((b) => {
    b.addEventListener("click", async () => {
      const busid = b.getAttribute("data-detach");
      await detachByBusid(busid);
    });
  });

  // bind auto toggle (VID:PID)
  el.querySelectorAll("input[type=checkbox][data-vidpid]").forEach((c) => {
    c.addEventListener("change", async () => {
      const vidpid = c.getAttribute("data-vidpid");
      const enabled = !!c.checked;
      await toggleAutoVidPid(vidpid, enabled);
    });
  });
}

async function refreshServersAndSettings() {
  const data = await api("/api/servers");
  state.servers = data.servers || [];
  state.selectedServerId = data.selectedServerId || (state.servers[0]?.id || "");  state.autoConnectIntervalSec = Number(data.autoConnectIntervalSec) || 5;
  renderServers();
  renderSettings();
}

async function refreshPorts() {
  try {
    const data = await api("/api/attached");
    state.ports = data.ports || [];

    const prev = new Set((state.attachedBusids || []).map(String));
    const next = new Set((data.ports || []).map(p => (p.remote_busid || p.local_busid || "").trim()).filter(Boolean).map(String));
    // Detect unexpected drops (e.g., remote server closed the connection)
    for (const b of prev) {
      if (!next.has(b)) {
        if (pendingUserDetaches.has(b)) { pendingUserDetaches.delete(b); }
        else { logK("log.portDropped", { busid: b }); scheduleExportsRefresh(250); }
      }
    }
    state.attachedBusids = [...next];
    renderPorts(state.ports);
    // Re-render exports UI using cached list so buttons reflect current attached state.
    if (state.exportsDevices && state.exportsDevices.length) renderExports(state.exportsDevices);
  } catch (e) {
    logK("log.portsError", { msg: e.message });
  }
}

async function refreshExports() {
  const srv = selectedServer();
  if (!srv) {
    renderExportsMessage("devices.noServer");
    return;
  }

  const host = srv.host || "";
  const port = Number(srv.tcpPort ?? 3240);

  // While we don't know, clear the list to avoid stale devices
  renderExportsMessage("devices.loading");
  setStatusKey("status.listingExports");

  try {
    const data = await api(`/api/exports`);
    state.exportsDevices = data.devices || [];
    renderExports(state.exportsDevices);
    logK("log.exports", { n: (data.devices || []).length, host: data.server, port: (data.tcpPort || 3240) });
    setStatusKey("status.ready");
  } catch (e) {
    // Clear list on unreachable / unknown state
    renderExportsMessage("devices.notConnected", { host, port });
    logK("log.remoteConnectFail", { host, port, msg: e.message });
    setStatusKey("status.error");
  }
}

async function refreshAll() {
  await refreshPorts();
  // realtime via ws
    
}

async function selectServer() {
  const id = $("serverSelect").value;
  if (!id) return;
  setStatusKey("status.selectingServer");
  renderExportsMessage("devices.loading");
  try {
    await api("/api/servers/select", { method: "POST", body: JSON.stringify({ id }) });
    state.selectedServerId = id;
    logK("log.serversSelected", { id });
    setStatusKey("status.ready");
    // Refresh when selecting server
    await refreshAll();
    await requestExports(true);
    restartAutoConnectLoop();
  } catch (e) {
    logK("log.serversError", { msg: e.message });
    setStatusKey("status.error");
  }
}

async function deleteServer() {
  const id = $("serverSelect").value;
  if (!id) return;
  setStatusKey("status.deletingServer");
  try {
    await api("/api/servers/delete", { method: "POST", body: JSON.stringify({ id }) });
    logK("log.serversDeleted", { id });
    setStatusKey("status.ready");
    await refreshServersAndSettings();
  await refreshAuthStatus();
    await refreshAll();
    if (!selectedServer()) renderExportsMessage("devices.noServer");
    await requestExports(true);
    restartAutoConnectLoop();
  } catch (e) {
    logK("log.serversError", { msg: e.message });
    setStatusKey("status.error");
  }
}

async function addServer() {
  const host = $("newHost").value.trim();
  const name = $("newName").value.trim();
  const tcpPortRaw = $("newPort").value.trim();
  const tcpPort = tcpPortRaw ? Number(tcpPortRaw) : 3240;
  if (!host) return;

  setStatusKey("status.addingServer");
  try {
    const data = await api("/api/servers/add", { method: "POST", body: JSON.stringify({ host, name, tcpPort }) });
    $("newHost").value = "";
    $("newName").value = "";
    $("newPort").value = "";
    state.servers = data.servers || [];
    state.selectedServerId = data.selectedServerId || state.selectedServerId;
    renderServers();
        const added = (data.servers && data.servers.length) ? data.servers[data.servers.length - 1] : null;
    logK("log.serversAdded", { host: (added?.host || host), id: (added?.id || state.selectedServerId || "") });
    setStatusKey("status.ready");
    // Refresh when adding server
    await refreshAll();
    await requestExports(true);
    restartAutoConnectLoop();
  } catch (e) {
    logK("log.serversError", { msg: e.message });
    setStatusKey("status.error");
  }
}

async function saveAuth() {
  const user = String($("authUser")?.value || "").trim();
  const pass = String($("authPass")?.value || "");

  const res = await api("/api/auth", { method: "POST", body: JSON.stringify({ user, pass }) });

  if (res.enabled) logK("log.authSaved", { user: res.user });
  else logK("log.authDisabled");

  if ($("authPass")) $("authPass").value = "";
}

async function refreshAuthStatus() {
  try {
    const res = await api("/api/auth", { method: "GET" });
    if ($("authUser")) $("authUser").value = res.user || "";
  } catch {
    // ignore
  }
}

async function saveSettings() {
  const interval = Number($("autoInterval").value);
  setStatusKey("status.savingInterval");
  try {
    const data = await api("/api/settings", { method: "POST", body: JSON.stringify({ autoConnectIntervalSec: interval }) });    state.autoConnectIntervalSec = Number(data.autoConnectIntervalSec) || 5;
    renderSettings();
    logK("log.settingsInterval", { sec: state.autoConnectIntervalSec });
    setStatusKey("status.ready");
    restartAutoConnectLoop();
  } catch (e) {
    logK("log.settingsError", { msg: e.message });
    setStatusKey("status.error");
  }
}

async function toggleAutoBusid(busid, enabled) {
  setStatusKey("status.updatingAuto");
  try {
    const data = await api("/api/servers/toggleAuto", { method: "POST", body: JSON.stringify({ busid, enabled }) });
    const srv = selectedServer();
    if (srv && srv.id === data.serverId) {
      srv.autoVidPids = data.autoVidPids || [];
      state.servers = state.servers.map(s => (s.id === srv.id ? srv : s));
    }
    logK("log.auto", { state: t(enabled ? "log.autoStateOn" : "log.autoStateOff"), vidpid: busid });
    setStatusKey("status.ready");
    restartAutoConnectLoop();
  } catch (e) {
    logK("log.autoError", { msg: e.message });
    setStatusKey("status.error");
  }
}

async function toggleAutoVidPid(vidpid, enabled) {
  const data = await api("/api/servers/toggleAuto", {
    method: "POST",
    body: JSON.stringify({ vidpid, enabled })
  });
  const sel = getSelectedServer();
  if (sel && data.autoVidPids) sel.autoVidPids = data.autoVidPids;
  if (window.__lastExports) renderExports(window.__lastExports);
}

async function attach(busid) {
  setStatusKey("status.connecting");
    const sel = getSelectedServer();
  const host = sel?.host || "";
  const port = Number(sel?.tcpPort ?? 3240);
  logK("log.attach", { busid, host, port });
  try {
    const data = await api("/api/attach", { method: "POST", body: JSON.stringify({ busid }) });
    if (data.alreadyAttached) logK("log.attachAlready", { busid });
    state.ports = data.ports || state.ports;
    renderPorts(state.ports);
    // Refresh on connect
    await refreshPorts();
    // realtime via ws
    
        logK("log.attachOk", { busid: (data.busid || busid) });
    setStatusKey("status.ready");
  } catch (e) {
    logK("log.attachError", { msg: e.message });
    setStatusKey("status.error");
  }
}

async function detachPort(port) {
  setStatusKey("status.disconnecting");
  logK("log.detachPort", { port });
  try {
    const data = await api("/api/detach", { method: "POST", body: JSON.stringify({ port }) });
    state.ports = data.ports || state.ports;
    renderPorts(state.ports);
    // Refresh on disconnect
    await refreshPorts();
    // realtime via ws
    
    logK("log.detachOk");
    setStatusKey("status.ready");
  } catch (e) {
    logK("log.detachError", { msg: e.message });
    setStatusKey("status.error");
  }
}

async function detachByBusid(busid) {
  try { pendingUserDetaches.add(String(busid)); } catch {}
  setStatusKey("status.disconnecting");
  logK("log.detachBusid", { busid });
  try {
    const data = await api("/api/detach", { method: "POST", body: JSON.stringify({ busid }) });
    state.ports = data.ports || state.ports;
    renderPorts(state.ports);
    // Refresh on disconnect
    await refreshPorts();
    // realtime via ws
    
    logK("log.detachOk");
    setStatusKey("status.ready");
  } catch (e) {
    logK("log.detachError", { msg: e.message });
    setStatusKey("status.error");
  }
}

// ------------------- Auto-connect loop -------------------
function restartAutoConnectLoop() { /* server-side enforcer handles auto-connect */ }

async function autoConnectTick() {
  // no-op (server-side enforcer does the work)
  return;
}


function startLocalPortPoll() {
  // Fallback polling only if WebSocket is not connected.
  setInterval(async () => {
    try {
      if (ws && ws.readyState === 1) return;
      await refreshPorts();
      if (window.__lastExports) renderExports(window.__lastExports);
    } catch {}
  }, 4000);
}

// ------------------- Bind UI -------------------
$("addServer").addEventListener("click", addServer);
$("selectServer").addEventListener("click", selectServer);
$("deleteServer").addEventListener("click", deleteServer);
$("refreshExports").addEventListener("click", () => refreshExports());
const _rp = document.getElementById("refreshPorts"); if (_rp) _rp.addEventListener("click", () => refreshPorts());
$("saveSettings").addEventListener("click", saveSettings);

$("saveAuth").addEventListener("click", saveAuth);

function wsSend(obj) {
  try {
    if (ws && ws.readyState === 1) {
      ws.send(JSON.stringify(obj));
      return true;
    }
  } catch {}
  return false;
}

async function requestExports(force = false) {
  const srv = selectedServer();
  if (!srv) {
    renderExportsMessage("devices.noServer");
    return;
  }

  // Clear stale list while waiting for WS/HTTP reply
  renderExportsMessage("devices.loading");

  // Prefer WS to keep UI instant; fallback to HTTP.
  if (wsSend({ type: "refreshExports", force })) return;
  await refreshExports();
}

window.addEventListener("load", async () => {
  setWsIndicator('off');
  setStatusKey('status.ready');
  connectWS();
  // i18n init (default: English)
  const savedLang = (() => { try { return localStorage.getItem("usbipWebUiLang"); } catch { return null; } })();
  setLang(savedLang || "en");
  const _ls = $("langSelect");
  if (_ls) _ls.addEventListener("change", () => setLang(_ls.value));
  log(t("log.started"));
  await refreshServersAndSettings();
  await refreshAll();
  restartAutoConnectLoop();
  startLocalPortPoll();
});
