import express from "express";
import { execFile } from "node:child_process";
import os from "node:os";
import path from "node:path";
import fs from "node:fs";
import http from "node:http";
import crypto from "node:crypto";
import { WebSocketServer } from "ws";

// ------------------- Config -------------------
// Defaults: bind localhost for safety. You can override via env.
const HOST = process.env.HOST || "127.0.0.1";
const PORT = Number(process.env.PORT || 8765);


// ------------------- Optional Auth (Basic) -------------------
// If credentials are configured, the Web UI/API requires HTTP Basic Auth.
// Credentials are independent from Linux users (no PAM).
// Configure via the Web UI (Settings -> Credentials) or by editing data/config.json.
const AUTH_CACHE_MS = 30_000;
const _authCache = new Map(); // key -> {ts, ok, user}

function ensureDefaultAuth(cfg) {
  // Force default login on first run (can be changed later).
  // Default: admin / usbip
  if (!cfg.auth || typeof cfg.auth !== "object") cfg.auth = { user: "", salt: "", hash: "" };
  if (cfg.auth.user && cfg.auth.salt && cfg.auth.hash) return false;

  const user = "admin";
  const pass = "usbip";
  const salt = crypto.randomBytes(16).toString("base64");
  const hash = scryptHash(pass, salt);
  cfg.auth = { user, salt, hash };
  return true;
}

function parseBasicAuth(req) {
  const h = req.headers["authorization"];
  if (!h || typeof h !== "string") return null;
  const m = h.match(/^Basic\s+(.+)$/i);
  if (!m) return null;
  try {
    const raw = Buffer.from(m[1], "base64").toString("utf8");
    const i = raw.indexOf(":");
    if (i < 0) return null;
    return { user: raw.slice(0, i), pass: raw.slice(i + 1) };
  } catch {
    return null;
  }
}

function scryptHash(pass, saltB64) {
  const salt = Buffer.from(String(saltB64 || ""), "base64");
  const out = crypto.scryptSync(String(pass || ""), salt, 32);
  return out.toString("base64");
}

function authConfigured(cfg) {
  const a = cfg?.auth || {};
  return !!(a.user && a.salt && a.hash);
}

async function checkAuth(req) {
  const cfg = loadConfig();
  if (!authConfigured(cfg)) return { ok: true, user: "" };

  const creds = parseBasicAuth(req);
  if (!creds?.user) return { ok: false, user: "" };

  const key = crypto.createHash("sha256").update(req.headers["authorization"] || "").digest("hex");
  const cached = _authCache.get(key);
  if (cached && (Date.now() - cached.ts) < AUTH_CACHE_MS) {
    return { ok: cached.ok, user: cached.user };
  }

  const a = cfg.auth;
  try {
    const userOk = (creds.user === a.user);
    const hash = scryptHash(creds.pass, a.salt);
    const hashOk = userOk && crypto.timingSafeEqual(Buffer.from(hash), Buffer.from(a.hash));
    _authCache.set(key, { ts: Date.now(), ok: !!hashOk, user: creds.user });
    return { ok: !!hashOk, user: creds.user };
  } catch {
    _authCache.set(key, { ts: Date.now(), ok: false, user: creds.user });
    return { ok: false, user: creds.user };
  }
}

function authMiddleware() {
  return async (req, res, next) => {
    try {
      const a = await checkAuth(req);
      if (a.ok) return next();
      res.setHeader("WWW-Authenticate", 'Basic realm="usbip-webui"');
      res.status(401).send("Authentication required");
    } catch {
      res.status(500).send("Auth error");
    }
  };
}


// Exports cache (server:port -> {ts, server, tcpPort, devices})
const exportsCache = new Map();
const EXPORTS_CACHE_MS = 5000;

// Config path (root service can override with USBIP_WEBUI_CONFIG=/var/lib/...)
const cfgPath = process.env.USBIP_WEBUI_CONFIG
  ? path.resolve(process.env.USBIP_WEBUI_CONFIG)
  : path.join(
      process.env.XDG_CONFIG_HOME || path.join(os.homedir(), ".config"),
      "usbip-webui-node-pro",
      "config.json"
    );

// ------------------- Rootless "surprise" mode -------------------
// If the UI server is not running as root, it can delegate privileged actions to the local agent
// over a UNIX socket (default: /run/usbip-webui/agent.sock).
const AGENT_SOCKET = process.env.USBIP_AGENT_SOCKET || "/run/usbip-webui/agent.sock";
const FORCE_AGENT = (process.env.USBIP_USE_AGENT || "").toLowerCase() === "1";

function shouldUseAgent() {
  const uid = process.getuid?.();
  if (uid !== undefined && uid !== 0) return true; // not root -> agent by default
  return FORCE_AGENT;
}

function agentRequest(method, pathname, body, timeoutMs = 20000) {
  return new Promise((resolve, reject) => {
    const payload = body ? Buffer.from(JSON.stringify(body)) : null;

    const req = http.request(
      {
        method,
        socketPath: AGENT_SOCKET,
        path: pathname,
        headers: payload
          ? { "Content-Type": "application/json", "Content-Length": payload.length }
          : undefined,
      },
      (res) => {
        let chunks = "";
        res.setEncoding("utf8");
        res.on("data", (d) => (chunks += d));
        res.on("end", () => {
          let data = {};
          try { data = chunks ? JSON.parse(chunks) : {}; } catch { data = { raw: chunks }; }
          if (res.statusCode && res.statusCode >= 200 && res.statusCode < 300) {
            resolve(data);
          } else {
            const msg = data?.detail || data?.raw || `Agent HTTP ${res.statusCode}`;
            reject(httpErr(500, msg));
          }
        });
      }
    );

    req.on("error", (e) => {
      if (e?.code === "ENOENT") {
        reject(httpErr(500, `Agent socket not found at ${AGENT_SOCKET}. Install/enable the agent service.`));
      } else if (e?.code === "EACCES") {
        reject(httpErr(500, `No permission to access agent socket (${AGENT_SOCKET}). Add your user to the usbipwebui group (or adjust permissions).`));
      } else {
        reject(httpErr(500, `Agent error: ${e?.message || e}`));
      }
    });

    req.setTimeout(timeoutMs, () => req.destroy(new Error("agent timeout")));

    if (payload) req.write(payload);
    req.end();
  });
}

async function agentHealth() {
  try {
    const h = await agentRequest("GET", "/health", null, 1500);
    return { ok: true, ...h };
  } catch (e) {
    return { ok: false, detail: e?.message || String(e) };
  }
}

// ------------------- Validation -------------------
const BUSID_RE = /^[0-9]+-[0-9]+(?:\.[0-9]+)*$/;

function httpErr(code, msg) {
  const e = new Error(msg);
  e.statusCode = code;
  return e;
}

function validateServer(s) {
  s = (s || "").trim();
  if (!s) throw httpErr(400, "server is required");
  // Conservative: IP (v4/v6 chars) or hostname (letters/digits/dot/dash)
  const isIpLike = /^[0-9a-fA-F:.]+$/.test(s);
  const isHostLike = /^[A-Za-z0-9.-]{1,253}$/.test(s) && !s.includes("..") && !s.startsWith("-") && !s.endsWith("-");
  if (!isIpLike && !isHostLike) throw httpErr(400, "Invalid server host/IP");
  return s;
}

function validateVidPid(vp) {
  const s = String(vp || "").trim().toLowerCase();
  if (!/^[0-9a-f]{4}:[0-9a-f]{4}$/.test(s)) throw httpErr(400, "Invalid VID:PID");
  return s;
}

function validateBusid(b) {
  b = (b || "").trim();
  if (!BUSID_RE.test(b)) throw httpErr(400, "Invalid busid (e.g. 1-1.1)");
  return b;
}

function validateTcpPort(p) {
  if (p === undefined || p === null || p === "") return 3240;
  const n = Number(p);
  if (!Number.isFinite(n) || n < 1 || n > 65535) throw httpErr(400, "Invalid tcpPort (1-65535)");
  return Math.trunc(n);
}

function validateIntervalSec(s) {
  const n = Number(s);
  // clamp: 5s .. 3600s
  if (!Number.isFinite(n) || n < 5 || n > 3600) throw httpErr(400, "Invalid autoConnectIntervalSec (5-3600)");
  return Math.trunc(n);
}

function validateName(n) {
  n = (n || "").trim();
  if (!n) return "";
  if (n.length > 40) throw httpErr(400, "name too long");
  if (!/^[\w .-]+$/.test(n)) throw httpErr(400, "Invalid name (allowed: letters, digits, space, . - _)");
  return n;
}

// ------------------- Config store -------------------
function loadConfig() {
  let changed = false;
  try {
    return normalizeConfig(JSON.parse(fs.readFileSync(cfgPath, "utf8")));
  } catch {
    return normalizeConfig({
      selectedServerId: "",
      // Global settings
      autoConnectEnabled: true,
      autoConnectIntervalSec: 5,
      servers: [] // {id, host, name, tcpPort, autoVidPids:[]}
    });
  }
}

function normalizeConfig(cfg) {
  cfg = cfg || {};
  let changed = false;

  if (!Array.isArray(cfg.servers)) { cfg.servers = []; changed = true; }
  if (typeof cfg.autoConnectEnabled !== "boolean") { cfg.autoConnectEnabled = true; changed = true; }

  const n = Number(cfg.autoConnectIntervalSec);
  const interval = (Number.isFinite(n) && n >= 5 && n <= 3600) ? Math.trunc(n) : 5;
  if (cfg.autoConnectIntervalSec !== interval) { cfg.autoConnectIntervalSec = interval; changed = true; }

  for (const s of cfg.servers) {
    if (!s.tcpPort) { s.tcpPort = 3240; changed = true; }
    if (!Array.isArray(s.autoVidPids)) { s.autoVidPids = []; changed = true; }
  }

  if (cfg.selectedServerId && !cfg.servers.some(s => s.id === cfg.selectedServerId)) {
    cfg.selectedServerId = cfg.servers[0]?.id || "";
    changed = true;
  }

  // Force default auth on first run
  if (ensureDefaultAuth(cfg)) changed = true;

  if (changed) {
    try { fs.mkdirSync(path.dirname(cfgPath), { recursive: true }); } catch {}
    try { fs.writeFileSync(cfgPath, JSON.stringify(cfg, null, 2)); } catch {}
  }

  return cfg;
}

function saveConfig(cfg) {
  cfg = normalizeConfig(cfg);
  fs.mkdirSync(path.dirname(cfgPath), { recursive: true });
  fs.writeFileSync(cfgPath, JSON.stringify(cfg, null, 2) + "\n", "utf8");
}

function newId() {
  return Math.random().toString(16).slice(2, 10) + Date.now().toString(16).slice(-6);
}

function getSelectedServer(cfg) {
  if (!cfg.selectedServerId) return null;
  return cfg.servers.find(s => s.id === cfg.selectedServerId) || null;
}

// ------------------- Command runner -------------------
// IMPORTANT: no shell. argv list only.
function runCmd(argv, timeoutMs = 20000) {
  return new Promise((resolve, reject) => {
    execFile(argv[0], argv.slice(1), { timeout: timeoutMs }, (err, stdout, stderr) => {
      if (err) {
        const msg = (String(stderr || stdout || err.message || "Command failed")).trim();
        reject(httpErr(500, msg));
        return;
      }
      resolve({ stdout: String(stdout), stderr: String(stderr) });
    });
  });
}

// ------------------- usbip parsing helpers -------------------
function usbipArgv(tcpPort, command, args) {
  // Keep compatibility with older usbip: only add --tcp-port when not default.
  const port = validateTcpPort(tcpPort);
  if (port && port !== 3240) {
    return ["usbip", "--tcp-port", String(port), command, ...args];
  }
  return ["usbip", command, ...args];
}

function parseUsbipList(out) {
  // Supports multiple usbip output variants, e.g.:
  //  " - 10.0.0.106"
  //  "      1-1.1: Silicon Labs : CP210x UART Bridge (10c4:ea60)"
  // or:
  //  " - 1-1.1: Vendor Product (046d:c534)"
  //
  // We parse any line that starts with a BUSID followed by ":" and extract VID:PID
  // if present at end in "(vvvv:pppp)".
  const devices = [];
  const seen = new Set();

  const lineRe = /^\s*(?:-\s*)?(?<busid>[0-9]+-[0-9]+(?:\.[0-9]+)*)\s*:\s*(?<rest>.*)$/;
  const vidpidRe = /\(\s*([0-9a-fA-F]{4})\s*:\s*([0-9a-fA-F]{4})\s*\)\s*$/;

  for (const line of out.split("\n")) {
    const m = line.match(lineRe);
    if (!m?.groups) continue;

    const busid = m.groups.busid;
    if (seen.has(busid)) continue;

    let rest = (m.groups.rest || "").trim();
    let vid = "";
    let pid = "";

    const vp = rest.match(vidpidRe);
    if (vp) {
      vid = (vp[1] || "").toLowerCase();
      pid = (vp[2] || "").toLowerCase();
      rest = rest.replace(vidpidRe, "").trim();
    }

    // Some outputs include trailing vendor/product decorations; keep as description
    devices.push({ busid, desc: rest, vid, pid });
    seen.add(busid);
  }
  return devices;
}

function parseUsbipPort(out) {
  // Robust parser for `usbip port` output (varies across distros).
  const ports = [];
  let cur = null;

  const portRe = /^Port\s+(?<port>\d+)\s*:\s*(?<status>.*)$/i;
  const uriRe = /usbip:\/\/(?<remote>[^\/\s]+)\/(?<busid>[0-9]+-[0-9]+(?:\.[0-9]+)*)/i;
  const mapRe = /^\s*(?<local>[0-9]+-[0-9]+(?:\.[0-9]+)*)\s*->\s*usbip:\/\/(?<remote>[^\/\s]+)\/(?<busid>[0-9]+-[0-9]+(?:\.[0-9]+)*)/i;

  for (const line of String(out || "").split("\n")) {
    const pm = line.match(portRe);
    if (pm?.groups) {
      if (cur) ports.push(cur);
      cur = {
        port: Number(pm.groups.port),
        status: (pm.groups.status || "").trim(),
        remote: "",
        remote_busid: "",
        local_busid: ""
      };
      continue;
    }
    if (!cur) continue;

    const mm = line.match(mapRe);
    if (mm?.groups) {
      cur.local_busid = (mm.groups.local || "").trim();
      cur.remote = (mm.groups.remote || "").trim();
      cur.remote_busid = (mm.groups.busid || "").trim();
      continue;
    }

    const um = line.match(uriRe);
    if (um?.groups) {
      cur.remote = (um.groups.remote || "").trim();
      cur.remote_busid = (um.groups.busid || "").trim();
      continue;
    }
  }

  if (cur) ports.push(cur);
  return ports;
}


async function usbipPortStdout() {
  if (shouldUseAgent()) {
    const data = await agentRequest("GET", "/usbip/port", null, 15000);
    return String(data.stdout || "");
  }
  const { stdout } = await runCmd(["usbip", "port"], 15000);
  return String(stdout || "");
}

async function usbipPorts() {
  const out = await usbipPortStdout();
  return parseUsbipPort(out);
}

async function resolvePortByBusid(busid) {
  busid = validateBusid(busid);
  const ports = await usbipPorts();
  const hit = ports.find(p => p.remote_busid === busid || p.local_busid === busid);
  if (!hit) throw httpErr(404, "Device not attached (busid not found in usbip port)");
  return hit.port;
}

// ------------------- Express app -------------------
const app = express();

app.use(authMiddleware());

app.use(express.json());
app.use(express.static("public"));

// Health
app.get("/api/health", async (req, res) => {
  const agent = await agentHealth();
  res.json({ ok: true, pid: process.pid, uid: process.getuid?.(), usingAgent: shouldUseAgent(), agentSocket: AGENT_SOCKET, agent });
});

// Servers CRUD
app.get("/api/servers", (req, res) => {
  const cfg = loadConfig();
  res.json({
    selectedServerId: cfg.selectedServerId || "",
    autoConnectEnabled: true,
    autoConnectIntervalSec: Number(cfg.autoConnectIntervalSec) || 5,
    servers: cfg.servers || []
  });
});


app.post("/api/servers/add", (req, res, next) => {
  try {
    const host = validateServer(req.body?.host);
    const name = validateName(req.body?.name || "");
    const tcpPort = validateTcpPort(req.body?.tcpPort ?? 3240);
    const cfg = loadConfig();
    const id = newId();
    cfg.servers = cfg.servers || [];
    cfg.servers.push({ id, host, name, tcpPort, autoVidPids: [] });
    if (!cfg.selectedServerId) cfg.selectedServerId = id;
    saveConfig(cfg);
        try { const cfg2 = loadConfig(); wsBroadcast({ type: "servers", servers: cfg2.servers || [], selectedServerId: cfg2.selectedServerId || "", autoConnectEnabled: true, autoConnectIntervalSec: Number(cfg2.autoConnectIntervalSec)||5 }); } catch {}
res.json({ ok: true, selectedServerId: cfg.selectedServerId, servers: cfg.servers });
  } catch (e) { next(e); }
});

app.post("/api/servers/delete", (req, res, next) => {
  try {
    const id = String(req.body?.id || "").trim();
    if (!id) throw httpErr(400, "id required");
    const cfg = loadConfig();
    cfg.servers = (cfg.servers || []).filter(s => s.id !== id);
    if (cfg.selectedServerId === id) cfg.selectedServerId = cfg.servers[0]?.id || "";
    saveConfig(cfg);
        try { const cfg2 = loadConfig(); wsBroadcast({ type: "servers", servers: cfg2.servers || [], selectedServerId: cfg2.selectedServerId || "", autoConnectEnabled: true, autoConnectIntervalSec: Number(cfg2.autoConnectIntervalSec)||5 }); } catch {}
res.json({ ok: true, selectedServerId: cfg.selectedServerId, servers: cfg.servers });
  } catch (e) { next(e); }
});

app.post("/api/servers/select", (req, res, next) => {
  try {
    const id = String(req.body?.id || "").trim();
    if (!id) throw httpErr(400, "id required");
    const cfg = loadConfig();
    const exists = (cfg.servers || []).some(s => s.id === id);
    if (!exists) throw httpErr(404, "server id not found");
    cfg.selectedServerId = id;
    saveConfig(cfg);
        try { const cfg2 = loadConfig(); wsBroadcast({ type: "servers", servers: cfg2.servers || [], selectedServerId: cfg2.selectedServerId || "", autoConnectEnabled: true, autoConnectIntervalSec: Number(cfg2.autoConnectIntervalSec)||5 }); } catch {}
res.json({ ok: true, selectedServerId: cfg.selectedServerId });
  } catch (e) { next(e); }
});
app.get("/api/auth", (req, res) => {
  const cfg = loadConfig();
  const enabled = authConfigured(cfg);
  res.json({ ok: true, enabled, user: enabled ? (cfg.auth.user || "") : "" });
});

app.post("/api/auth", async (req, res, next) => {
  try {
    const cfg = loadConfig();
    const configured = authConfigured(cfg);

    // If already configured, require current auth to change it
    if (configured) {
      const a = await checkAuth(req);
      if (!a.ok) {
        res.setHeader("WWW-Authenticate", 'Basic realm="usbip-webui"');
        res.status(401).send("Authentication required");
        return;
      }
    }

    const user = String(req.body?.user || "").trim();
    const pass = String(req.body?.pass || "");

    if (!user) throw httpErr(400, "User required");

    if (pass.length < 4) throw httpErr(400, "Password too short (min 4)");

    const salt = crypto.randomBytes(16).toString("base64");
    const hash = scryptHash(pass, salt);
    cfg.auth = { user, salt, hash };
    saveConfig(cfg);
    _authCache.clear();
    res.json({ ok: true, enabled: true, user });
  } catch (e) { next(e); }
});

app.post("/api/settings", (req, res, next) => {
  try {
    const cfg = loadConfig();
        if (req.body?.autoConnectIntervalSec !== undefined) {
      cfg.autoConnectIntervalSec = validateIntervalSec(req.body.autoConnectIntervalSec);
    }
    saveConfig(cfg);
        try { const cfg2 = loadConfig(); wsBroadcast({ type: "servers", servers: cfg2.servers || [], selectedServerId: cfg2.selectedServerId || "", autoConnectEnabled: true, autoConnectIntervalSec: Number(cfg2.autoConnectIntervalSec)||5 }); } catch {}
    try { startAutoEnforcer(); } catch {}
res.json({
      ok: true,
      autoConnectEnabled: true,
      autoConnectIntervalSec: Number(cfg.autoConnectIntervalSec) || 5
    });
  } catch (e) { next(e); }
});

app.post("/api/servers/toggleAuto", async (req, res, next) => {
  try {
    const cfg = loadConfig();
    const sel = getSelectedServer(cfg);
    if (!sel) throw httpErr(400, "No server selected. Add one first.");

    const vidpid = validateVidPid(req.body?.vidpid);
    const enabled = !!req.body?.enabled;

    sel.autoVidPids = Array.isArray(sel.autoVidPids) ? sel.autoVidPids.map(String).map(v => v.toLowerCase()).filter(Boolean) : [];
    const has = sel.autoVidPids.includes(vidpid);
    if (enabled && !has) sel.autoVidPids.push(vidpid);
    if (!enabled && has) sel.autoVidPids = sel.autoVidPids.filter(v => v !== vidpid);

    cfg.servers = (cfg.servers || []).map(s => (s.id === sel.id ? sel : s));
    saveConfig(cfg);

    // If user enabled auto for this VID:PID, try to attach immediately (best effort).
    if (enabled) {
      try {
        const exp = await getExportsSelected({ force: false });
        const portOut = await usbipPortStdout();
        const cand = (exp.devices || []).find(d => `${d.vid}:${d.pid}`.toLowerCase() === vidpid && !portOutHasBusid(portOut, d.busid));
        if (cand) await doAttachInternal(exp.server, exp.tcpPort, cand.busid);
      } catch {}
    }

    try { startAutoEnforcer(); } catch {}
    try { const cfg2 = loadConfig(); wsBroadcast({ type: "servers", servers: cfg2.servers || [], selectedServerId: cfg2.selectedServerId || "", autoConnectEnabled: true, autoConnectIntervalSec: Number(cfg2.autoConnectIntervalSec)||5 }); } catch {}

    res.json({ ok: true, serverId: sel.id, autoVidPids: sel.autoVidPids });
  } catch (e) { next(e); }
});






// Export list from selected server (or explicit server)
async function getExportsFor(server, tcpPort, { force = false } = {}) {
  const key = `${server}:${tcpPort}`;
  if (!force) {
    const cached = exportsCache.get(key);
    if (cached && (Date.now() - cached.ts) < EXPORTS_CACHE_MS) return cached;
  }

  let stdout = "";
  if (shouldUseAgent()) {
    const data = await agentRequest("POST", "/usbip/list", { host: server, tcpPort }, 25000);
    stdout = String(data.stdout || "");
  } else {
    const argv = usbipArgv(tcpPort, "list", ["--remote", server]);
    ({ stdout } = await runCmd(argv, 25000));
  }

  const devices = parseUsbipList(stdout);
  const value = { ts: Date.now(), server, tcpPort, devices };
  exportsCache.set(key, value);
  return value;
}

async function getExportsSelected({ force = false } = {}) {
  const cfg = loadConfig();
  const sel = getSelectedServer(cfg);
  if (!sel) throw httpErr(400, "No server selected. Add one first.");

  const server = sel.host;
  const tcpPort = validateTcpPort(sel.tcpPort ?? 3240);

  const exp = await getExportsFor(server, tcpPort, { force });
  const devices = exp.devices || [];

  // One-time migration: legacy BUSID-based auto list -> VID:PID
  try {
    if (Array.isArray(sel.autoBusids) && sel.autoBusids.length) {
      sel.autoVidPids = Array.isArray(sel.autoVidPids) ? sel.autoVidPids : [];
      const map = new Map(devices.map(d => [d.busid, `${d.vid}:${d.pid}`.toLowerCase()]));
      for (const b of sel.autoBusids.map(String)) {
        const vp = map.get(b);
        if (vp && /^[0-9a-f]{4}:[0-9a-f]{4}$/.test(vp) && !sel.autoVidPids.includes(vp)) sel.autoVidPids.push(vp);
      }
      delete sel.autoBusids;
      cfg.servers = (cfg.servers || []).map(s => (s.id === sel.id ? sel : s));
      saveConfig(cfg);
    }
  } catch {}

  return { ts: exp.ts, server, tcpPort, devices };
}


app.get("/api/exports", async (req, res, next) => {
  try {
    const cfg = loadConfig();
    const serverOverride = (req.query.server || "").toString().trim();
    const tcpPortOverride = (req.query.tcpPort || "").toString().trim();
    let server = "";
    let tcpPort = 3240;

    if (serverOverride) {
      server = validateServer(serverOverride);
      tcpPort = tcpPortOverride ? validateTcpPort(tcpPortOverride) : 3240;
    } else {
      const sel = getSelectedServer(cfg);
      if (!sel) throw httpErr(400, "No server selected. Add one first.");
      server = sel.host;
      tcpPort = validateTcpPort(sel.tcpPort ?? 3240);
    }

    let stdout = "";

    if (shouldUseAgent()) {
      const data = await agentRequest("POST", "/usbip/list", { host: server, tcpPort }, 25000);
      stdout = String(data.stdout || "");
    } else {
      const argv = usbipArgv(tcpPort, "list", ["--remote", server]);
      ({ stdout } = await runCmd(argv, 25000));
    }

    const devices = parseUsbipList(stdout);
        exportsCache.set(`${server}:${tcpPort}`, { ts: Date.now(), server, tcpPort, devices });
    try { wsBroadcast({ type: "exports", server, tcpPort, devices }); } catch {}
    res.json({ server, tcpPort, devices });
  } catch (e) { next(e); }
});



// Attached ports (local)
app.get("/api/attached", async (req, res, next) => {
  try {
    const ports = await usbipPorts();
    res.json({ ports });
  } catch (e) { next(e); }
});

// Attach (uses selected server unless overridden)
app.post("/api/attach", async (req, res, next) => {
  try {
    const cfg = loadConfig();
    const serverOverride = (req.body?.server || "").toString().trim();
    const tcpPortOverride = (req.body?.tcpPort || "").toString().trim();

    let server = "";
    let tcpPort = 3240;

    if (serverOverride) {
      server = validateServer(serverOverride);
      tcpPort = tcpPortOverride ? validateTcpPort(tcpPortOverride) : 3240;
    } else {
      const sel = getSelectedServer(cfg);
      if (!sel) throw httpErr(400, "No server selected. Add one first.");
      server = sel.host;
      tcpPort = validateTcpPort(sel.tcpPort ?? 3240);
    }

    const busid = validateBusid(req.body?.busid);

        // Guard: do not allow attaching the same remote BUSID more than once.
    // Hard safety net to avoid duplicate ports under auto-connect.
    const portOut = await usbipPortStdout();
    const currentPorts = parseUsbipPort(portOut);
    
    // Fast raw check (survives parser differences)
    const rawHasBusid = portOut.includes(`/${busid}`) || portOut.includes(`${busid} ->`) || portOut.includes(` ${busid}:`);
    
    const alreadyAttached = rawHasBusid || currentPorts.some((p) => {
      const b = (p.remote_busid || p.local_busid || "").trim();
      if (!b || b !== busid) return false;
      const r = String(p.remote || "");
      // If remote is known, ensure it matches the selected server (avoid false positives across servers).
      return !r || r.includes(server);
    });
    
    if (alreadyAttached) {
      try { wsBroadcast({ type: "ports", ports: currentPorts, attachedBusids: Array.from(new Set(currentPorts.map(p => (p.remote_busid || p.local_busid || "").trim()).filter(Boolean))) }); } catch {}
      res.json({ ok: true, server, tcpPort, busid, ports: currentPorts, alreadyAttached: true });
      return;
    }


    if (shouldUseAgent()) {
      const data = await agentRequest("POST", "/usbip/attach", { host: server, busid, tcpPort }, 30000);
      const ports = parseUsbipPort(String(data.stdout || ""));
      try { wsBroadcast({ type: "ports", ports, attachedBusids: Array.from(new Set(ports.map(p => (p.remote_busid || p.local_busid || "").trim()).filter(Boolean))) }); } catch {}
      res.json({ ok: true, server, tcpPort, busid, ports });
      return;
    }

    // Root mode (legacy)
    await runCmd(["modprobe", "vhci_hcd"], 10000).catch(() => {});
    const argv = usbipArgv(tcpPort, "attach", ["--remote", server, "--busid", busid]);
    await runCmd(argv, 30000);
    const ports = await usbipPorts();
    try { wsBroadcast({ type: "ports", ports, attachedBusids: Array.from(new Set(ports.map(p => (p.remote_busid || p.local_busid || "").trim()).filter(Boolean))) }); } catch {}
    res.json({ ok: true, server, tcpPort, busid, ports });
  } catch (e) { next(e); }
});

app.post("/api/detach", async (req, res, next) => {
  try {
    const portRaw = req.body?.port;
    const busidRaw = (req.body?.busid || "").toString().trim();

    let port = null;
    if (Number.isFinite(Number(portRaw))) port = Number(portRaw);
    if (port === null && !busidRaw) throw httpErr(400, "Provide port or busid");

    if (port === null && busidRaw) {
      // Uses agent transparently for port listing in rootless mode
      port = await resolvePortByBusid(busidRaw);
    }

    if (!Number.isFinite(Number(port)) || port < 0 || port > 255) {
      throw httpErr(400, "Invalid port");
    }

    if (shouldUseAgent()) {
      const data = await agentRequest("POST", "/usbip/detach", { port: Math.trunc(port) }, 30000);
      const ports = parseUsbipPort(String(data.stdout || ""));
          try { wsBroadcast({ type: "ports", ports, attachedBusids: Array.from(new Set(ports.map(p => (p.remote_busid || p.local_busid || "").trim()).filter(Boolean))) }); } catch {}
res.json({ ok: true, ports });
      return;
    }

    await runCmd(["usbip", "detach", "--port", String(port)], 20000);
    const ports = await usbipPorts();
    res.json({ ok: true, ports });
  } catch (e) { next(e); }
});


// Error handler
app.use((err, req, res, next) => {
  const code = err.statusCode || 500;
  res.status(code).json({ detail: err.message || "Error" });
});


// ------------------- Server-side auto-connect enforcer -------------------
// Runs even when no browser UI is open.
let _autoTimer = null;
let _autoTimerIntervalSec = null;

function stopAutoEnforcer() {
  if (_autoTimer) clearInterval(_autoTimer);
  _autoTimer = null;
  _autoTimerIntervalSec = null;
}

function startAutoEnforcer() {
  const cfg = loadConfig();
  const interval = Number(cfg.autoConnectIntervalSec) || 5;

  const sec = Math.max(5, Math.min(3600, Math.trunc(interval)));
  if (_autoTimer && _autoTimerIntervalSec === sec) return;

  stopAutoEnforcer();
  _autoTimerIntervalSec = sec;
  wsLog(`[auto] enforcer active (every ${sec}s)`);

  _autoTimer = setInterval(() => {
    autoEnforcerTick().catch((e) => console.log("[auto][tick][error]", e?.message || e));
  }, sec * 1000);

  // small initial delay
  setTimeout(() => autoEnforcerTick().catch(() => {}), 800);
}

async function doAttachInternal(server, tcpPort, busid) {
  if (shouldUseAgent()) {
    await agentRequest("POST", "/usbip/attach", { host: server, tcpPort, busid }, 30000);
    return;
  }
  const argv = usbipArgv(tcpPort, "attach", ["--remote", server, "--busid", busid]);
  await runCmd(argv, 30000);
}

function portOutHasBusid(portOut, busid) {
  const s = String(portOut || "");
  return s.includes(`/${busid}`) || s.includes(`${busid} ->`) || s.includes(` ${busid}:`);
}

async function autoEnforcerTick() {
  const cfg = loadConfig();

  const servers = Array.isArray(cfg.servers) ? cfg.servers : [];
  const targets = [];
  for (const s of servers) {
    // Backward-compat: migrate legacy BUSID auto list -> VID:PID (works even if Web UI is not opened)
    if ((!Array.isArray(s.autoVidPids) || !s.autoVidPids.length) && Array.isArray(s.autoBusids) && s.autoBusids.length) {
      try {
        const exp = await getExportsFor(s.host, Number(s.tcpPort) || 3240, { force: false });
        const map = new Map((exp.devices || []).map(d => [d.busid, `${d.vid}:${d.pid}`.toLowerCase()]));
        s.autoVidPids = Array.isArray(s.autoVidPids) ? s.autoVidPids : [];
        for (const b of s.autoBusids.map(String)) {
          const vp = map.get(b);
          if (vp && /^[0-9a-f]{4}:[0-9a-f]{4}$/.test(vp) && !s.autoVidPids.includes(vp)) s.autoVidPids.push(vp);
        }
        delete s.autoBusids;
        saveConfig(cfg);
      } catch {}
    }
    const vps = Array.isArray(s.autoVidPids) ? s.autoVidPids.map(String).map(v => v.toLowerCase()).filter(Boolean) : [];
    const filtered = vps.filter(v => /^[0-9a-f]{4}:[0-9a-f]{4}$/.test(v));
    if (!filtered.length) continue;
    targets.push({ host: s.host, tcpPort: Number(s.tcpPort) || 3240, vidpids: filtered });
  }
  if (!targets.length) return;

  const portOut = await usbipPortStdout();
  const ports = parseUsbipPort(portOut);

  for (const t of targets) {
    // get exports mapping for this server
    let exp = null;
    try { exp = await getExportsFor(t.host, t.tcpPort, { force: false }); } catch {}
    const devices = exp?.devices || [];

    const busidToVidpid = new Map();
    for (const d of devices) {
      const vp = `${d.vid}:${d.pid}`.toLowerCase();
      if (/^[0-9a-f]{4}:[0-9a-f]{4}$/.test(vp)) busidToVidpid.set(d.busid, vp);
    }

    // collect attached busids for this server
    const attachedBusids = new Set();
    for (const p of ports) {
      const b = (p.remote_busid || p.local_busid || "").trim();
      if (!b) continue;
      const r = String(p.remote || "");
      if (!r || r.includes(t.host)) attachedBusids.add(b);
    }

    const attachedVidpids = new Set();
    for (const b of attachedBusids) {
      const vp = busidToVidpid.get(b);
      if (vp) attachedVidpids.add(vp);
    }

    for (const vp of t.vidpids) {
      if (attachedVidpids.has(vp)) continue;

      const cand = devices.find(d => `${d.vid}:${d.pid}`.toLowerCase() === vp && !portOutHasBusid(portOut, d.busid));
      if (!cand) continue;

      try {
        await doAttachInternal(t.host, t.tcpPort, cand.busid);
        wsLog(`[auto] attached ${vp} (${cand.busid}) -> ${t.host}:${t.tcpPort}`);
      } catch (e) {
        wsLog(`[auto][error] attach ${vp} (${cand.busid}) -> ${t.host}:${t.tcpPort}:`, e?.message || e);
      }
    }
  }
}

// ------------------- HTTP + WebSocket server -------------------
const httpServer = app.listen(PORT, HOST, () => {
  startAutoEnforcer();
  console.log("USB/IP Web UI iniciado.");
  console.log(`config: ${cfgPath}`);
  try {
    const cfg = loadConfig();
    if (authConfigured(cfg)) {
      console.log(`auth: enabled (basic) · default: admin/usbip`);
    }
  } catch {}

  console.log(`Web UI: http://127.0.0.1:${PORT}`);
});

const wss = new WebSocketServer({ noServer: true });

httpServer.on("upgrade", async (req, socket, head) => {
  try {
    if (req.url !== "/ws") return;

    // Auth check for WS (only enforced if credentials are configured)
    const a = await checkAuth(req);
    if (!a.ok) {
      socket.write('HTTP/1.1 401 Unauthorized\r\nWWW-Authenticate: Basic realm="usbip-webui"\r\n\r\n');
      socket.destroy();
      return;
    }

    wss.handleUpgrade(req, socket, head, (ws) => {
      wss.emit("connection", ws, req);
    });
  } catch {
    try { socket.destroy(); } catch {}
  }
});


function wsBroadcast(obj) {
  const msg = JSON.stringify(obj);
  for (const client of wss.clients) {
    if (client.readyState === 1) client.send(msg);
  }
}

function wsLog(line) {
  console.log(line);
  wsBroadcast({ type: "log", line });
}

function portsSignature(ports) {
  const flat = (ports || []).map(p => ({
    port: Number(p.port),
    status: String(p.status || ""),
    remote: String(p.remote || ""),
    remote_busid: String(p.remote_busid || ""),
    local_busid: String(p.local_busid || "")
  }));
  return JSON.stringify(flat);
}

let lastPortsSig = "";

async function getPortsAndAttachedBusids() {
  const ports = await usbipPorts();
  const s = new Set();
  for (const p of ports) {
    const b = (p.remote_busid || p.local_busid || "").trim();
    if (b) s.add(b);
  }
  return { ports, attachedBusids: [...s] };
}

async function wsSendState(ws) {
  const cfg = loadConfig();
  const sel = getSelectedServer(cfg);
  const { ports, attachedBusids } = await getPortsAndAttachedBusids();
  ws.send(JSON.stringify({
    type: "state",
    servers: cfg.servers || [],
    selectedServerId: cfg.selectedServerId || "",
    autoConnectEnabled: true,
    autoConnectIntervalSec: Number(cfg.autoConnectIntervalSec) || 5,
    selectedServer: sel ? { id: sel.id, host: sel.host, name: sel.name, tcpPort: sel.tcpPort, autoVidPids: sel.autoVidPids || [] } : null,
    attachedBusids,
    ports
  }));
}

wss.on("connection", async (ws, req) => {
  try {
    await wsSendState(ws);
    try {
      const exp = await getExportsSelected({ force: false });
      ws.send(JSON.stringify({ type: "exports", server: exp.server, tcpPort: exp.tcpPort, devices: exp.devices }));
    } catch (e) {
    }
  } catch (e) {
    try { console.warn("[ws][error]", String(e?.message || e)); } catch {}
  }

  ws.on("message", async (buf) => {
    let msg = null;
    try { msg = JSON.parse(String(buf || "")); } catch {}
    if (msg?.type === "getState") {
      try {
        await wsSendState(ws);
        const exp = await getExportsSelected({ force: false });
        ws.send(JSON.stringify({ type: "exports", server: exp.server, tcpPort: exp.tcpPort, devices: exp.devices }));
      } catch {}
      return;
    }
    if (msg?.type === "refreshExports") {
      try {
        const exp = await getExportsSelected({ force: !!msg.force });
        ws.send(JSON.stringify({ type: "exports", server: exp.server, tcpPort: exp.tcpPort, devices: exp.devices }));
      } catch (e) {
        try {
          const cfgNow = loadConfig();
          const sel = getSelectedServer(cfgNow);
          ws.send(JSON.stringify({
            type: "exportsError",
            msg: String(e?.message || e),
            server: sel?.host || "",
            tcpPort: sel?.tcpPort || 3240
          }));
        } catch {}
        try { ws.send(JSON.stringify({ type: "log", line: `[exports][error] ${String(e?.message || e)}` })); } catch {}
      }
      return;
    }
  });
});

// Local-only port watcher to update UI quickly when kernel drops a connection.
setInterval(async () => {
  try {
    const ports = await usbipPorts();
    const sig = portsSignature(ports);
    if (sig !== lastPortsSig) {
      lastPortsSig = sig;
      const s = new Set();
      for (const p of ports) {
        const b = (p.remote_busid || p.local_busid || "").trim();
        if (b) s.add(b);
      }
      wsBroadcast({ type: "ports", ports, attachedBusids: [...s] });
    }
  } catch {}
}, 1500);

